﻿

namespace Hims.Api.Controllers
{
    using Hims.Api.Models;
    using Hims.Domain.Entities;
    using Hims.Domain.Helpers;
    using Hims.Domain.Services;
    using Hims.Infrastructure.Services;
    using Hims.Shared.DataFilters;
    using Hims.Shared.EntityModels;
    using Hims.Shared.Library.Enums;
    using Hims.Shared.UserModels.Filters;
    using Hims.Shared.UserModels.WebNotification;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/contactdetails")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ContactDetailsController :BaseController
    {
        /// <summary>
        /// The coupon services.
        /// </summary>
        private readonly IContactDetailsService contactDetailsService;
        private readonly IAESHelper aesHelper;

        /// <inheritdoc />
        public ContactDetailsController(IContactDetailsService contactDetailsService, IAuditLogService auditLogServices, IAESHelper aesHelper)
        {
            this.contactDetailsService = contactDetailsService;
            this.aesHelper = aesHelper;
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<ContactDetails>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] ContactDetails model, [FromHeader] LocationHeader header)
        {
            model = (ContactDetails)EmptyFilter.Handler(model);
            var beds = await this.contactDetailsService.FetchAsync(model);
            return beds == null ? this.ServerError() : this.Success(beds);
        }

        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon added successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] ContactDetails model, [FromHeader] LocationHeader header)
        {
            model = (ContactDetails)EmptyFilter.Handler(model);
            var response = await this.contactDetailsService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given ContactDetails has already been exists with us.");
                case 0:
                    return this.ServerError();
            }
            return this.Success("ContactDetails has been added successfully.");
        }

        /// <summary>
        /// The delete specialization.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Specialization deleted successfully.
        /// - 409 - Specialization can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] ContactDetails model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (ContactDetails)EmptyFilter.Handler(model);
                var response = await this.contactDetailsService.DeleteAsync(model.Id);
                if (response == 0)
                {
                    return this.ServerError();
                }


                return this.Success("ContactDetails has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                if (exception.Message.Contains("violates foreign key constraint"))
                {
                    return this.Conflict("ContactDetails can't be deleted. Please contact Administrator.");
                }

                return this.ServerError();
            }
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetchbyId")]
        [ProducesResponseType(typeof(List<ContactDetails>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindProblemById([FromBody] ContactDetailsModel model)
        {
            model.PatientId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedPatientId));
            model = (ContactDetailsModel)EmptyFilter.Handler(model);
            var wards = await this.contactDetailsService.FindById(model.PatientId);
            return wards == null ? this.ServerError() : this.Success(wards);
        }
        /// <summary>
        /// The update coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon updated successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] ContactDetails model, [FromHeader] LocationHeader header)
        {
            model = (ContactDetails)EmptyFilter.Handler(model);
            var response = await this.contactDetailsService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given ContactDetails has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            return this.Success("ContactDetails has been updated successfully.");
        }
    }
}
